'use client'

import { useEffect, useState } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Formulario, RespuestaFormulario } from '@/lib/types'
import Link from 'next/link'

export default function FormularioDetailPage() {
  const params = useParams()
  const router = useRouter()
  const [formulario, setFormulario] = useState<Formulario | null>(null)
  const [respuestas, setRespuestas] = useState<RespuestaFormulario[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (params.id) {
      fetchFormulario(params.id as string)
      fetchRespuestas(params.id as string)
    }
  }, [params.id])

  async function fetchFormulario(id: string) {
    try {
      const { data, error } = await supabase
        .from('formularios')
        .select('*')
        .eq('id', id)
        .single()

      if (error) throw error
      setFormulario(data)
    } catch (error) {
      console.error('Error fetching formulario:', error)
    } finally {
      setLoading(false)
    }
  }

  async function fetchRespuestas(formularioId: string) {
    try {
      const { data, error } = await supabase
        .from('respuestas_formularios')
        .select('*')
        .eq('formulario_id', formularioId)
        .order('created_at', { ascending: false })

      if (error) throw error
      setRespuestas(data || [])
    } catch (error) {
      console.error('Error fetching respuestas:', error)
    }
  }

  async function copiarLink() {
    if (!formulario) return
    const linkCompleto = `${window.location.origin}/formularios/publico/${formulario.link_publico}`
    try {
      await navigator.clipboard.writeText(linkCompleto)
      alert('Link copiado al portapapeles!')
    } catch (error) {
      console.error('Error copiando link:', error)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  if (!formulario) {
    return (
      <div className="text-center p-8">
        <p>Formulario no encontrado</p>
        <Link href="/formularios" className="text-blue-600 hover:underline">Volver a formularios</Link>
      </div>
    )
  }

  const linkCompleto = `${typeof window !== 'undefined' ? window.location.origin : ''}/formularios/publico/${formulario.link_publico}`

  return (
    <div className="max-w-4xl mx-auto">
      <div className="mb-4">
        <Link href="/formularios" className="text-blue-600 hover:underline">← Volver a formularios</Link>
      </div>
      
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <div className="flex justify-between items-start mb-4">
          <div>
            <h1 className="text-3xl font-bold mb-2">{formulario.titulo}</h1>
            {formulario.descripcion && (
              <p className="text-gray-600">{formulario.descripcion}</p>
            )}
          </div>
          <span className={`px-3 py-1 rounded text-sm ${
            formulario.activo ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
          }`}>
            {formulario.activo ? 'Activo' : 'Inactivo'}
          </span>
        </div>

        <div className="bg-blue-50 p-4 rounded-lg mb-4">
          <p className="text-sm font-medium mb-2">Link público del formulario:</p>
          <div className="flex gap-2">
            <input
              type="text"
              value={linkCompleto}
              readOnly
              className="flex-1 border rounded px-3 py-2 bg-white text-sm"
            />
            <button
              onClick={copiarLink}
              className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
            >
              Copiar
            </button>
          </div>
        </div>

        <div>
          <h2 className="text-xl font-bold mb-4">Campos del Formulario</h2>
          <div className="space-y-2">
            {formulario.campos?.map((campo, index) => (
              <div key={campo.id} className="border rounded p-3 bg-gray-50">
                <div className="flex items-center gap-2">
                  <span className="text-sm font-medium">{index + 1}.</span>
                  <span className="font-semibold">{campo.etiqueta}</span>
                  <span className="text-xs text-gray-500">({campo.tipo})</span>
                  {campo.requerido && (
                    <span className="text-xs bg-red-100 text-red-700 px-2 py-1 rounded">Requerido</span>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      <div className="bg-white p-6 rounded-lg shadow">
        <h2 className="text-xl font-bold mb-4">Respuestas ({respuestas.length})</h2>
        {respuestas.length === 0 ? (
          <p className="text-gray-500">Aún no hay respuestas para este formulario.</p>
        ) : (
          <div className="space-y-4">
            {respuestas.map((respuesta) => (
              <div key={respuesta.id} className="border rounded p-4 bg-gray-50">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <p className="text-sm text-gray-500">
                      Enviado: {new Date(respuesta.created_at).toLocaleString('es-ES')}
                    </p>
                    {respuesta.fecha_visita && (
                      <p className="text-sm text-gray-500">
                        Fecha de visita: {new Date(respuesta.fecha_visita).toLocaleDateString('es-ES')}
                      </p>
                    )}
                  </div>
                </div>
                <div className="space-y-2">
                  {Object.entries(respuesta.datos || {}).map(([key, value]) => (
                    <div key={key} className="border-l-2 border-blue-500 pl-3">
                      <p className="text-sm font-medium text-gray-700">{key}:</p>
                      <p className="text-sm text-gray-900">{String(value)}</p>
                    </div>
                  ))}
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
